<?php
require_once '../config.php';
verificarLogin();

$titulo = "Editar Cliente";
$security = new Security($pdo);

// Buscar planos DO USUÁRIO ATUAL
$stmt = $pdo->prepare("SELECT * FROM planos WHERE usuario_id = ? ORDER BY nome");
$stmt->execute([$_SESSION['usuario_id']]);
$planos = $stmt->fetchAll();

// Buscar servidores DO USUÁRIO ATUAL
$stmt = $pdo->prepare("SELECT * FROM servidores WHERE status = 'ativo' AND usuario_id = ? ORDER BY nome");
$stmt->execute([$_SESSION['usuario_id']]);
$servidores = $stmt->fetchAll();

// Buscar cliente DO USUÁRIO ATUAL
$id = $_GET['id'];
$stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = ? AND usuario_id = ?");
$stmt->execute([$id, $_SESSION['usuario_id']]);
$cliente = $stmt->fetch();

if (!$cliente) {
    header("Location: clientes.php");
    exit();
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verificar CSRF
    if (!$security->verifyCSRFToken($_POST['csrf_token'])) {
        die("Token CSRF inválido!");
    }
    
    $nome = $security->cleanInput($_POST['nome']);
    $telefone = $security->cleanInput($_POST['telefone']);
    $plano_id = $security->cleanInput($_POST['plano_id']);
    $servidor_id = $_POST['servidor_id'] ? $security->cleanInput($_POST['servidor_id']) : null;
    $data_vencimento = $security->cleanInput($_POST['data_vencimento']);
    $status = $security->cleanInput($_POST['status']);
    $observacoes = $security->cleanInput($_POST['observacoes']);
    
    // Validar dados
    $errors = [];
    
    if (!$security->validatePhone($telefone)) {
        $errors[] = "Telefone inválido!";
    }
    
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("UPDATE clientes SET nome = ?, telefone = ?, plano_id = ?, servidor_id = ?, data_vencimento = ?, status = ?, observacoes = ? WHERE id = ? AND usuario_id = ?");
            $stmt->execute([$nome, $telefone, $plano_id, $servidor_id, $data_vencimento, $status, $observacoes, $id, $_SESSION['usuario_id']]);
            
            // Registrar na auditoria
            $stmt = $pdo->prepare("INSERT INTO security_logs (usuario_id, acao, ip_address, user_agent) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['usuario_id'], 'editar_cliente', $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
            
            header("Location: clientes.php?sucesso=1");
            exit();
        } catch (PDOException $e) {
            $erro = "Erro ao atualizar cliente: " . $e->getMessage();
        }
    } else {
        $erro = implode("<br>", $errors);
    }
}

$csrf_token = $security->generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $titulo; ?> - Gestor DREEYTV</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        :root {
            --primary: #667eea;
            --secondary: #2d3748;
            --danger: #f56565;
        }
        
        body {
            background: #f5f7fa;
            color: #2d3748;
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .sidebar {
            width: 280px;
            background: var(--secondary);
            color: white;
        }
        
        .sidebar-header {
            padding: 30px 25px;
            background: rgba(0,0,0,0.2);
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .sidebar-header h2 {
            font-size: 1.5rem;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .sidebar-menu {
            list-style: none;
            padding: 20px 0;
        }
        
        .sidebar-menu li {
            margin: 5px 15px;
        }
        
        .sidebar-menu a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.8);
            text-decoration: none;
            border-radius: 10px;
            transition: all 0.3s;
        }
        
        .sidebar-menu a:hover {
            background: rgba(255,255,255,0.1);
            color: white;
        }
        
        .sidebar-menu a.active {
            background: var(--primary);
            color: white;
        }
        
        .main-content {
            flex: 1;
            padding: 30px;
        }
        
        .header {
            background: white;
            padding: 25px 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 1.8rem;
            background: linear-gradient(135deg, var(--primary), #5a6fd8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), #5a6fd8);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
        }
        
        .content {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            max-width: 600px;
            margin: 0 auto;
        }
        
        .form-header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .form-header .icon {
            font-size: 3rem;
            margin-bottom: 10px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #2d3748;
        }
        
        .form-control {
            width: 100%;
            padding: 15px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s;
        }
        
        .form-control:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        textarea.form-control {
            min-height: 100px;
            resize: vertical;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 15px 25px;
            background: var(--primary);
            color: white;
            text-decoration: none;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s;
            border: none;
            cursor: pointer;
            font-size: 16px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .btn-block {
            display: block;
            width: 100%;
        }
        
        .btn-danger {
            background: var(--danger);
        }
        
        .btn-danger:hover {
            box-shadow: 0 5px 15px rgba(245, 101, 101, 0.3);
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        
        .alert {
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .alert-danger {
            background: #fed7d7;
            color: #742a2a;
            border: 1px solid #feb2b2;
        }
        
        @media (max-width: 768px) {
            .admin-container {
                flex-direction: column;
            }
            
            .sidebar {
                width: 100%;
            }
            
            .form-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>📺 GESTOR DREEYTV</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="../index.php">📊 Dashboard</a></li>
                <li><a href="clientes.php">👨‍💼 Clientes</a></li>
                <li><a href="../planos/planos.php">📋 Planos</a></li>
                <li><a href="../servidores/servidores.php">🖥️ Servidores</a></li>
                <?php if ($_SESSION['usuario_tipo'] == 'admin'): ?>
                    <li><a href="../usuarios/usuarios.php">👥 Usuários</a></li>
                <?php endif; ?>
                <li><a href="../senha/alterar_senha.php">🔑 Alterar Senha</a></li>
                <li><a href="../logout.php">🚪 Sair</a></li>
            </ul>
        </div>
        
        <div class="main-content">
            <div class="header">
                <h1>Editar Cliente</h1>
                <div class="user-info">
                    <div class="user-avatar"><?php echo substr($_SESSION['usuario_nome'], 0, 1); ?></div>
                    <div>
                        <div style="font-weight: 600;"><?php echo $_SESSION['usuario_nome']; ?></div>
                        <div style="font-size: 0.9rem; color: #666;"><?php echo ucfirst($_SESSION['usuario_tipo']); ?></div>
                    </div>
                    <a href="../logout.php" class="btn" style="background: var(--danger);">🚪 Sair</a>
                </div>
            </div>
            
            <div class="content">
                <div class="form-header">
                    <div class="icon">✏️</div>
                    <h2>Editar Cliente</h2>
                    <p style="color: #666;">Atualize os dados do cliente</p>
                </div>
                
                <?php if (isset($erro)): ?>
                    <div class="alert alert-danger">
                        <strong>❌ Erro:</strong> <?php echo $erro; ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                    
                    <div class="form-group">
                        <label class="form-label" for="nome">👤 Nome Completo</label>
                        <input type="text" id="nome" name="nome" class="form-control" value="<?php echo htmlspecialchars($cliente['nome']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="telefone">📞 Telefone/WhatsApp</label>
                        <input type="text" id="telefone" name="telefone" class="form-control" value="<?php echo htmlspecialchars($cliente['telefone']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="plano_id">📋 Plano</label>
                        <select id="plano_id" name="plano_id" class="form-control" required>
                            <?php foreach ($planos as $plano): ?>
                                <option value="<?php echo $plano['id']; ?>" <?php echo $plano['id'] == $cliente['plano_id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($plano['nome']); ?> - 
                                    R$ <?php echo number_format($plano['valor'], 2, ',', '.'); ?> - 
                                    <?php echo $plano['dias']; ?> dias
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="servidor_id">🖥️ Servidor (Opcional)</label>
                        <select id="servidor_id" name="servidor_id" class="form-control">
                            <option value="">Sem servidor</option>
                            <?php foreach ($servidores as $servidor): ?>
                                <option value="<?php echo $servidor['id']; ?>" <?php echo $servidor['id'] == $cliente['servidor_id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($servidor['nome']); ?> - 
                                    <?php echo htmlspecialchars($servidor['tipo']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="data_vencimento">📅 Data de Vencimento</label>
                        <input type="date" id="data_vencimento" name="data_vencimento" class="form-control" value="<?php echo $cliente['data_vencimento']; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="status">🔍 Status</label>
                        <select id="status" name="status" class="form-control" required>
                            <option value="ativo" <?php echo $cliente['status'] == 'ativo' ? 'selected' : ''; ?>>✅ Ativo</option>
                            <option value="inativo" <?php echo $cliente['status'] == 'inativo' ? 'selected' : ''; ?>>❌ Inativo</option>
                            <option value="pendente" <?php echo $cliente['status'] == 'pendente' ? 'selected' : ''; ?>>⏳ Pendente</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="observacoes">📝 Observações</label>
                        <textarea id="observacoes" name="observacoes" class="form-control" placeholder="Anotações sobre o cliente..."><?php echo htmlspecialchars($cliente['observacoes']); ?></textarea>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-block">💾 Atualizar Cliente</button>
                        <a href="clientes.php" class="btn btn-danger btn-block">❌ Cancelar</a>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Máscara para telefone
        document.getElementById('telefone').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length <= 11) {
                value = value.replace(/(\d{2})(\d)/, '($1) $2');
                value = value.replace(/(\d{5})(\d)/, '$1-$2');
            }
            e.target.value = value;
        });
    </script>
</body>
</html>